const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const { body, validationResult } = require('express-validator');
const nodemailer = require('nodemailer');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 5000;

// Middleware de seguridad
app.use(helmet());

// Configuración CORS
app.use(cors({
  origin: process.env.FRONTEND_URL || 'http://localhost:3000',
  credentials: true,
  methods: ['GET', 'POST'],
  allowedHeaders: ['Content-Type', 'Authorization']
}));

// Middleware para parsear JSON
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Rate limiting - máximo 5 emails por IP cada 15 minutos
const emailLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutos
  max: 5, // máximo 5 requests por ventana de tiempo
  message: {
    error: 'Demasiados intentos de envío de email. Intenta nuevamente en 15 minutos.',
    code: 'RATE_LIMIT_EXCEEDED'
  },
  standardHeaders: true,
  legacyHeaders: false
});

// Configuración del transportador de email
const createEmailTransporter = () => {
  return nodemailer.createTransport({
    host: process.env.EMAIL_HOST,
    port: parseInt(process.env.EMAIL_PORT),
    secure: process.env.EMAIL_SECURE === 'true',
    auth: {
      user: process.env.EMAIL_USER,
      pass: process.env.EMAIL_PASS
    },
    tls: {
      rejectUnauthorized: false
    }
  });
};

// Validaciones para el formulario de contacto
const contactValidation = [
  body('nombre')
    .trim()
    .isLength({ min: 2, max: 100 })
    .withMessage('El nombre debe tener entre 2 y 100 caracteres')
    .matches(/^[a-zA-ZáéíóúÁÉÍÓÚñÑ\s]+$/)
    .withMessage('El nombre solo puede contener letras y espacios'),
  
  body('email')
    .isEmail()
    .normalizeEmail()
    .withMessage('Debe proporcionar un email válido'),
  
  body('telefono')
    .optional()
    .trim()
    .matches(/^[+]?[0-9\s\-()]{7,20}$/)
    .withMessage('El teléfono debe tener un formato válido'),
  
  body('empresa')
    .optional()
    .trim()
    .isLength({ max: 100 })
    .withMessage('El nombre de la empresa no puede exceder 100 caracteres'),
  
  body('mensaje')
    .trim()
    .isLength({ min: 10, max: 1000 })
    .withMessage('El mensaje debe tener entre 10 y 1000 caracteres')
];

// Ruta de salud del servidor
app.get('/health', (req, res) => {
  res.status(200).json({
    status: 'OK',
    message: 'Servidor funcionando correctamente',
    timestamp: new Date().toISOString(),
    environment: process.env.NODE_ENV
  });
});

// Endpoint para envío de emails de contacto
app.post('/api/contact', emailLimiter, contactValidation, async (req, res) => {
  try {
    // Verificar errores de validación
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'Datos de entrada inválidos',
        errors: errors.array()
      });
    }

    const { nombre, email, telefono, empresa, mensaje } = req.body;

    // Crear transportador de email
    const transporter = createEmailTransporter();

    // Verificar conexión con el servidor de email
    await transporter.verify();

    // Configurar el email para SynchroDev
    const mailOptions = {
      from: `"${nombre}" <${process.env.EMAIL_USER}>`,
      to: ['rgonzalezco@synchrodev.cl', 'jhuerta@synchrodev.cl', 'smardones@synchrodev.cl', 'nvega@synchrodev.cl'],
      subject: `Nuevo mensaje de contacto desde synchrodev.cl - ${nombre}`,
      html: `<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; background-color: #f8fafc; border-radius: 10px;">
          <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
            <h1 style="color: white; margin: 0; font-size: 24px;">Nuevo Mensaje de Contacto</h1>
            <p style="color: #e2e8f0; margin: 10px 0 0 0;">synchrodev.cl</p>
          </div>
          
          <div style="background: white; padding: 30px; border-radius: 0 0 10px 10px; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
            <div style="margin-bottom: 20px;">
              <h3 style="color: #2d3748; margin: 0 0 15px 0; border-bottom: 2px solid #e2e8f0; padding-bottom: 10px;">Información del Cliente</h3>
              
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px 0; vertical-align: top;">
                    <table style="width: 100%; border-collapse: collapse;">
                      <tr>
                        <td style="width: 40px; vertical-align: top; padding-right: 10px;">
                          <div style="background: #667eea; color: white; padding: 8px; border-radius: 50%; width: 24px; height: 24px; text-align: center; line-height: 24px; font-size: 10px; font-weight: bold;">N</div>
                        </td>
                        <td style="vertical-align: top;">
                          <strong style="color: #2d3748;">Nombre:</strong><br>
                          <span style="color: #4a5568;">${nombre}</span>
                        </td>
                      </tr>
                    </table>
                  </td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; vertical-align: top;">
                    <table style="width: 100%; border-collapse: collapse;">
                      <tr>
                        <td style="width: 40px; vertical-align: top; padding-right: 10px;">
                          <div style="background: #667eea; color: white; padding: 8px; border-radius: 50%; width: 24px; height: 24px; text-align: center; line-height: 24px; font-size: 10px; font-weight: bold;">@</div>
                        </td>
                        <td style="vertical-align: top;">
                          <strong style="color: #2d3748;">Email:</strong><br>
                          <a href="mailto:${email}" style="color: #667eea; text-decoration: none;">${email}</a>
                        </td>
                      </tr>
                    </table>
                  </td>
                </tr>
                
                ${telefono ? `
                <tr>
                  <td style="padding: 8px 0; vertical-align: top;">
                    <table style="width: 100%; border-collapse: collapse;">
                      <tr>
                        <td style="width: 40px; vertical-align: top; padding-right: 10px;">
                          <div style="background: #667eea; color: white; padding: 8px; border-radius: 50%; width: 24px; height: 24px; text-align: center; line-height: 24px; font-size: 10px; font-weight: bold;">T</div>
                        </td>
                        <td style="vertical-align: top;">
                          <strong style="color: #2d3748;">Teléfono:</strong><br>
                          <a href="tel:${telefono}" style="color: #667eea; text-decoration: none;">${telefono}</a>
                        </td>
                      </tr>
                    </table>
                  </td>
                </tr>
                ` : ''}
                
                ${empresa ? `
                <tr>
                  <td style="padding: 8px 0; vertical-align: top;">
                    <table style="width: 100%; border-collapse: collapse;">
                      <tr>
                        <td style="width: 40px; vertical-align: top; padding-right: 10px;">
                          <div style="background: #667eea; color: white; padding: 8px; border-radius: 50%; width: 24px; height: 24px; text-align: center; line-height: 24px; font-size: 10px; font-weight: bold;">E</div>
                        </td>
                        <td style="vertical-align: top;">
                          <strong style="color: #2d3748;">Empresa:</strong><br>
                          <span style="color: #4a5568;">${empresa}</span>
                        </td>
                      </tr>
                    </table>
                  </td>
                </tr>
                ` : ''}
              </table>
            </div>
            
            <div style="margin-top: 25px;">
              <h3 style="color: #2d3748; margin: 0 0 15px 0; border-bottom: 2px solid #e2e8f0; padding-bottom: 10px;">Mensaje</h3>
              <div style="background: #f7fafc; padding: 20px; border-radius: 8px; border-left: 4px solid #667eea;">
                <p style="color: #4a5568; line-height: 1.6; margin: 0; white-space: pre-wrap;">${mensaje}</p>
              </div>
            </div>
            
            <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #e2e8f0; text-align: center;">
              <p style="color: #718096; font-size: 14px; margin: 0;">Mensaje recibido el ${new Date().toLocaleString('es-CL', { timeZone: 'America/Santiago' })}</p>
              <p style="color: #a0aec0; font-size: 12px; margin: 5px 0 0 0;">SynchroDev - Sincronizando las nuevas tecnologías</p>
            </div>
          </div>
        </div>
      `,
      text: `Nuevo mensaje de contacto desde synchrodev.cl

Información del cliente:
- Nombre: ${nombre}
- Email: ${email}
${telefono ? `- Teléfono: ${telefono}` : ''}
${empresa ? `- Empresa: ${empresa}` : ''}

Mensaje:
${mensaje}

Recibido el: ${new Date().toLocaleString('es-CL', { timeZone: 'America/Santiago' })}`
    };

    // Configurar email de confirmación para el cliente
    const confirmationMailOptions = {
      from: `"SynchroDev" <${process.env.EMAIL_USER}>`,
      to: email,
      subject: 'Confirmación de mensaje recibido - SynchroDev',
      html: `<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; background-color: #f8fafc; border-radius: 10px;">
          <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 30px; border-radius: 10px 10px 0 0; text-align: center;">
            <h1 style="color: white; margin: 0; font-size: 24px;">¡Gracias por contactarnos!</h1>
            <p style="color: rgba(255, 255, 255, 0.9); margin: 10px 0 0 0; font-weight: 500;">SynchroDev</p>
          </div>
          
          <div style="background: white; padding: 30px; border-radius: 0 0 10px 10px; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
            <p style="color: #2d3748; font-size: 16px; line-height: 1.6;">Hola <strong>${nombre}</strong>,</p>
            
            <p style="color: #4a5568; line-height: 1.6;">Hemos recibido tu mensaje y queremos agradecerte por contactarnos. Nuestro equipo revisará tu consulta y te responderemos a la brevedad.</p>
            
            <div style="background: #f7fafc; padding: 20px; border-radius: 8px; border-left: 4px solid #667eea; margin: 20px 0;">
              <h3 style="color: #2d3748; margin: 0 0 10px 0;">Resumen de tu mensaje:</h3>
              <p style="color: #4a5568; margin: 0; white-space: pre-wrap;">${mensaje}</p>
            </div>
            
            <p style="color: #4a5568; line-height: 1.6;">Mientras tanto, te invitamos a seguirnos en nuestras redes sociales para estar al día con nuestros proyectos y novedades.</p>
            
            <div style="text-align: center; margin: 30px 0;">
              <a href="https://www.linkedin.com/company/synchrodev" style="display: inline-block; margin: 0 10px; padding: 10px 20px; background: #0077b5; color: white; text-decoration: none; border-radius: 5px;">LinkedIn</a>
              <a href="https://www.instagram.com/synchrodevcl" style="display: inline-block; margin: 0 10px; padding: 10px 20px; background: #e4405f; color: white; text-decoration: none; border-radius: 5px;">Instagram</a>
            </div>
            
            <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #e2e8f0; text-align: center;">
              <p style="color: #718096; font-size: 14px; margin: 0;">¡Gracias por confiar en SynchroDev!</p>
              <p style="color: #a0aec0; font-size: 12px; margin: 5px 0 0 0;">Sincronizando las nuevas tecnologías</p>
            </div>
          </div>
        </div>
      `,
      text: `¡Gracias por contactarnos!

Hola ${nombre},

Hemos recibido tu mensaje y queremos agradecerte por contactarnos. Nuestro equipo revisará tu consulta y te responderemos a la brevedad.

Resumen de tu mensaje:
${mensaje}

¡Gracias por confiar en SynchroDev!
Sincronizando las nuevas tecnologías`
    };

    // Enviar ambos emails
    await Promise.all([
      transporter.sendMail(mailOptions),
      transporter.sendMail(confirmationMailOptions)
    ]);

    // Respuesta exitosa
    res.status(200).json({
      success: true,
      message: 'Mensaje enviado exitosamente. Te responderemos pronto.',
      timestamp: new Date().toISOString()
    });

  } catch (error) {
    console.error('Error al enviar email:', error);
    
    // Respuesta de error
    res.status(500).json({
      success: false,
      message: 'Error interno del servidor. Por favor, intenta nuevamente más tarde.',
      error: process.env.NODE_ENV === 'development' ? error.message : undefined
    });
  }
});

// Middleware para rutas no encontradas
app.use('*', (req, res) => {
  res.status(404).json({
    success: false,
    message: 'Ruta no encontrada',
    path: req.originalUrl
  });
});

// Middleware de manejo de errores global
app.use((error, req, res, next) => {
  console.error('Error no manejado:', error);
  
  res.status(500).json({
    success: false,
    message: 'Error interno del servidor',
    error: process.env.NODE_ENV === 'development' ? error.message : undefined
  });
});

// Iniciar servidor
app.listen(PORT, () => {
  console.log(`🚀 Servidor ejecutándose en puerto ${PORT}`);
  console.log(`🌍 Entorno: ${process.env.NODE_ENV}`);
  console.log(`📧 Servidor de email: ${process.env.EMAIL_HOST}:${process.env.EMAIL_PORT}`);
  console.log(`🔗 CORS habilitado para: ${process.env.FRONTEND_URL}`);
});

// Manejo de cierre graceful
process.on('SIGTERM', () => {
  console.log('🛑 Recibida señal SIGTERM, cerrando servidor...');
  process.exit(0);
});

process.on('SIGINT', () => {
  console.log('🛑 Recibida señal SIGINT, cerrando servidor...');
  process.exit(0);
});

module.exports = app;